/**
* @author Vclav Mikolek
* nicklaus@students.zcu.cz
*/

package animace.label;

import animace.*;
import javax.swing.*;
import javax.imageio.*;
import java.awt.*;
import java.awt.image.*;
import java.awt.event.*;
import java.io.*;
import java.util.*;


/**
 * Tda MoveablePicture poskytuje ukzku, jak by mli vypadat 
 * grafick komponenty pro tdu animace.label.PictureLabel.
 * Tedy komponenty pro systm, kde se programtor star o vykreslovn sm.
 * Podstatn je princip vykreslovn. Objekt pri njak zmn (pohyb) nastav
 * promnou needUpdate na true a vol repaint(). Metoda repaint() je volna s parametry
 * oznaujc tzv. dirty region. Takto by mla bt tato metoda volna pokad, jinak dochz asto zbyten
 * k pekreslovni cel obrazovky (panelu).
 * I kdy nen poteba objekt pekreslit, protoe na nm nedolo ke zmn (needUpdate nastaveno na false),
 * musme jet otestovat, zda jej jin objekt "nepejel", 
 * to se zjist z Clip oblasti nastaven v Graphics g. Viz ne metoda paint(Graphics g).
 * Dle stoj za povimnuti, e je implementovno pidvn poslucha udalost stejn, jako jsme zvykl
 * u klasickch komponent, tedy metodou add"Nco"Listener, v tomto ppad addActionListener. To je proto,
 * e lze oekvat, e se na objekty bude klikat. V komponent, na kter tyto objekty vis, je nutn jet
 * zajistit, e bude volat metodu processEvent(...) nad tou komponentou, na kterou uivatel kliknul.
 * Je toho dost co musme napsat, abychom zajistili zkladn funknost.
 */
public class MoveablePicture implements Moveable, PaintControled {
	private Component owner;
	private BufferedImage img;
	private PaintControl pc = null;
	private ActionListener actionListener = null;
	
	/**
	 * Indikuje, e na objetku nastaly zmny, a e je nutn jej znovu pekreslit.
	 */
	public boolean needUpdate = true;
		
	private int x = 0;		// location
	private int y = 0;		// location
	private int imgWidth = 0;
	private int imgHeight = 0;
		
	/**
	 * Konstruktor vyaduje jako parametr grafickho vlastnika (rodie) objektu,
	 * tedy nejakou Componentu na ktere spov.
	 * @param owner Componenta, na kter je MoveablePicture nakreslen
	 * @param img vlastn obrzek, kter se bude pohybovat
	 */
	public MoveablePicture(Component owner,BufferedImage img) {
		this.owner = owner;
		this.img = img;
		imgWidth = img.getWidth();
		imgHeight = img.getHeight();
	}

	/**
	 * Nakresl tento objekt do grafickho kontextu pedanho parametrem.
	 * Algortimus je nsledujc. Pokud se s objektem hbalo a je teba jej pekreslit,
	 * needUpdate == true, pak nic nezmeme a musme jej kreslit. Ale pokud needUpdate == false a 
	 * tento objekt nen v "dirty regionu" (g.getClipBounds()) kreslit jej nemusme a uetme as.
	 */
	public void paint(Graphics g) {
		if (!needUpdate) {
			Rectangle toDraw = g.getClipBounds().intersection(new Rectangle(x,y,imgWidth,imgHeight));
			if (!toDraw.isEmpty()) {
				g.drawImage(img,x,y,imgWidth,imgHeight,null);
			}
		}
		else {
			g.drawImage(img,x,y,imgWidth,imgHeight,null);
			needUpdate = false;
		}
	}
	
	/**
	 * Vrac obrzek
	 */
	public BufferedImage getImage() {
		return img;
	}
	
	/**
	 * Vrac preferovanou velikost - rozmry obrzku.
	 */
	public Dimension getPrefferedSize(){
		return new Dimension(img.getWidth(),img.getHeight());
	}

	/**
	 * Vrac ku obrzku
	 */
	public int getWidth() {
		return img.getWidth();
	}

	/**
	 * Vrac vku obrzku
	 */
	public int getHeight() 
	{
		return img.getHeight();
	}
	
	/**
	 * Metoda z rozhran Moveable. Nastav novou pozici objektu a hodnotu needUpdate na true.
	 * Pot zavol owner.repaint(int x, int y, int width, int height), kter
	 * zaregistruje poadavek na pekreslen pouze t sti obrazovky,
     * kde dolo ke zmnm na grafice.
	 * Pi zavolani tto metody, nedochz k dnm kontrolm,
	 * zda jsou daje sprvn
	 */
	public synchronized void setLocation(int x,int y) {
		int left = (this.x < x ? this.x : x) - 1;
		int up = (this.y < y ? this.y : y) - 1;
		int dx = Math.abs(this.x - x) + 2;
		int dy = Math.abs(this.y - y) + 2;
		this.x = x;
		this.y = y;
		needUpdate = true;
		owner.repaint(left,up,imgWidth+dx,imgHeight+dy);
	}
	
	/**
	 * Metoda z rozhrani Moveable, Funguje standardn, vrac pozici objektu
	 */
	public Point getLocation(){
		return new Point(x,y);
	}
		
	/**
	 * Metoda z rozhran PaintControled
	 */
	public void setPaintControl(PaintControl pc) {
		this.pc = pc;
	}
	
	/**
	 * Metoda z rozhran PaintControled
	 */
	public PaintControl getPaintControl() {
		return pc;
	}
	
	/**
	 * Metoda addActionListener, jak jsme zvykl z java.AWT.Component
	 * Zde implementovna pomoc tdy AWTEventMulticaster.
	 */
	public void addActionListener(ActionListener newActionListener) {
		actionListener = AWTEventMulticaster.add(actionListener, newActionListener);
	}
	
	/** 
	 * Metoda processEvent, kterou bude volat zejmna pedek objektu,
	 * v naem ppad PictureLabel, provede zavoln metody actionPerformed(...) nad vemi
	 * zaregistrovanmi posluchai.
	 */
	public void processEvent(AWTEvent e) {
		if (actionListener != null) {
			actionListener.actionPerformed(new ActionEvent(this,0,""));
		}
	}
	
	/**
	 * Vrac true, jestlie objekt obsahuje urit bod, hodi se pri testu na kliknuti
	 * @param point bod, kter je testovn, zda le na obrazku
	 */
	public boolean contains(Point point) {
		return (new Rectangle(x,y,imgWidth,imgHeight)).contains(point);
	}
		
}

